
/************************************************************************
 *
 * \file: DeviceMonitor.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: Android Auto - Demo application
 *
 * \author: J. Harder / ADITG/SW1 / jharder@de.adit-jv.com
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#include <sstream>
#include <aauto_macros.h>
#include <adit_logging.h>
#include <aauto/AautoLogging.h>
#include <aauto/AoapTransport.h>
#include "DeviceMonitor.h"

LOG_IMPORT_CONTEXT(demo)

namespace adit { namespace aauto {

using namespace uspi;

DeviceMonitor::DeviceMonitor()
{
    running = false;
    discoverer = nullptr;
    testLocation = "";

    pthread_mutex_init(&resetMutex, NULL);
}

DeviceMonitor::~DeviceMonitor()
{
    stop();
    pthread_mutex_destroy(&resetMutex);
}

bool DeviceMonitor::start()
{
    running = true;

    discoverer = new FeatureDiscovery(this, DD_USB_AOAP);

    /* Start Discoverer
     * Start monitoring of mobile device which supporting AOA protocl (AOAP) */
    if (DiscoveryError::OK != discoverer->start())
    {
        LOG_ERROR((demo, "failed to start FeatureDiscovery"));
        discoverer = nullptr;
        return false;
    }
    else
    {
        LOG_INFO((demo, "Discoverer started"));
    }

    return true;
}

void DeviceMonitor::stop()
{
    if (discoverer != nullptr)
    {
        if (DiscoveryError::OK != discoverer->stop())
        {
            LOG_ERROR((demo, "failed to properly stop FeatureDiscovery"));
            // just continue though
        }
        delete discoverer;
        discoverer = nullptr;
    }
}

void DeviceMonitor::waitForExit()
{
    // TODO what to do if discoverer is null?

    std::unique_lock<std::mutex> _lock(mMutex);
    mCondVarWaitForExit.wait(_lock);

    if (discoverer != nullptr) {
        if (resetFlag.test_and_set()) // quickly check non-blocking whether reset is available
        {
            // move list content to local copy so that we don't block while resetting
            pthread_mutex_lock(&resetMutex);
            auto infos = resetInfos;
            resetInfos.clear();
            pthread_mutex_unlock(&resetMutex);

            for (auto& id : infos)
            {
                DeviceInfo ddInfo = id->getInfo();

                LOG_INFO((demo, "DeviceMonitor::%s()  Call discoverer->resetDevice(%s) size=%zu",
                        __func__, to_string(ddInfo).c_str(), infos.size()));
                waitForDeviceReset(id);
            }
        }
    } else {
        LOG_ERROR((demo, "DeviceMonitor::%s()  No discoverer", __func__));
    }
}

void DeviceMonitor::requestStop()
{
    // set to false to avoid continuous testing while stopping
    running = false;

    std::lock_guard<std::mutex> guard(mMutex);
    mCondVarWaitForExit.notify_one();
}

void DeviceMonitor::waitForDeviceReset(std::shared_ptr<DiscoveredDeviceUsb> inUsbDevice)
{
    DiscoveryError res = DiscoveryError::INCOMPLETE;
    int32_t cnt = 0;
    DeviceInfo ddInfo = inUsbDevice->getInfo();

    while ((res == DiscoveryError::INCOMPLETE) && (cnt < 8))
    {
        res = inUsbDevice->resetDevice(FD_PROTOCOL_GOOGLE_AOAP, 1000);
        if (DiscoveryError::INCOMPLETE == res) {
            LOG_INFO((demo, "waitForDeviceReset() unable to reset device: %s, res = %s",
                     to_string(ddInfo).c_str(), to_str(res) ));
        } else if (DiscoveryError::OK != res) {
            LOG_WARN((demo, "waitForDeviceReset() reset device: %s failed, res = %s",
                     to_string(ddInfo).c_str(), to_str(res) ));
            break;
        } else {
        }
        cnt++;
    }
}

void DeviceMonitor::resetDevice(std::shared_ptr<DiscoveredDeviceUsb> inUsbDevice)
{
    // TODO t_usbDeviceInformation should not be used as we do not know if copy is safe
    pthread_mutex_lock(&resetMutex);
    resetInfos.push_back(inUsbDevice);
    pthread_mutex_unlock(&resetMutex);
    resetFlag.clear();
}

void DeviceMonitor::resetDevice(std::shared_ptr<DiscoveredDeviceUsb> inUsbDevice, bool testmode)
{
    (void)testmode;

    waitForDeviceReset(inUsbDevice);
}

DiscoveryError DeviceMonitor::foundCb(std::shared_ptr<DiscoveredDevice> inDevice)
{
    std::shared_ptr<DiscoveredDeviceUsb> pUsbDevice = std::dynamic_pointer_cast<DiscoveredDeviceUsb>(inDevice);

    DeviceInfo ddInfo = pUsbDevice->getInfo();
    LOG_INFO((demo, "device %s found", to_string(ddInfo).c_str() ));

    if (getRunningStatus()) {
        if (true == pUsbDevice->checkDevice(FD_PROTOCOL_GOOGLE_AOAP, 1000)) {
            LOG_INFO((demo, "device supports AOAP"));

            onDeviceFound(pUsbDevice);
        } else {
            LOG_INFO((demo, "device does not support AOAP"));
        }
    }

    return DiscoveryError::OK;
}

DiscoveryError DeviceMonitor::lostCb(std::shared_ptr<DiscoveredDevice> inDevice)
{
    std::shared_ptr<DiscoveredDeviceUsb> pUsbDevice = std::dynamic_pointer_cast<DiscoveredDeviceUsb>(inDevice);

    DeviceInfo ddInfo = pUsbDevice->getInfo();
    LOG_INFO((demo, "device %s removed", to_string(ddInfo).c_str()));

    if (getRunningStatus()) {
        onDeviceLost(pUsbDevice);
    }

    return DiscoveryError::OK;
}

DiscoveryError DeviceMonitor::switchedCb(std::shared_ptr<DiscoveredDevice> inDevice)
{
    std::shared_ptr<DiscoveredDeviceUsb> pUsbDevice = std::dynamic_pointer_cast<DiscoveredDeviceUsb>(inDevice);

    DeviceInfo ddInfo = pUsbDevice->getInfo();
    LOG_INFO((demo, "device %s in accessory mode found", to_string(ddInfo).c_str() ));

    onDeviceSwitched(pUsbDevice);

    return DiscoveryError::OK;
}

DiscoveryError DeviceMonitor::changedCb(std::shared_ptr<DiscoveredDevice> inDevice)
{
    std::shared_ptr<DiscoveredDeviceUsb> pUsbDevice = std::dynamic_pointer_cast<DiscoveredDeviceUsb>(inDevice);

    DeviceInfo ddInfo = pUsbDevice->getInfo();
    LOGD_DEBUG((demo, "device %s changed", to_string(ddInfo).c_str() ));

    onDeviceChanged(pUsbDevice);

    return DiscoveryError::OK;
}

DiscoveryError DeviceMonitor::errorCb(adit::uspi::DiscoveryError inErrorCode)
{
    LOG_INFO((demo, "Error %s occurred", to_str(inErrorCode) ));

    onDeviceError(inErrorCode);

    return DiscoveryError::OK;
}

std::string to_string(DeviceInfo& inVal)
{
    std::stringstream ss;
    ss << "vendorId=0x" << std::hex << inVal.getiDeviceInfo(DSYS_IDVENDOR) <<
          " productId=0x" << std::hex << inVal.getiDeviceInfo(DSYS_IDPRODUCT) <<
          " serial=" << inVal.getDeviceInfo(DSYS_SERIAL);
    return ss.str();
}

} } // namespace adit { namespace aauto {
